<?php
/*
Plugin Name: 终极图片粘贴
Description: 完美解决WordPress经典编辑器图片粘贴问题
Version: 3.3
Author: 小武
*/

defined('ABSPATH') || exit;

class Ultimate_Image_Paste {

    private $options;

    public function __construct() {
        $this->options = get_option('ultimate_image_paste_options', [
            'quality' => 85,
            'max_width' => 1600,
            'max_height' => 1600,
            'enable_paste' => true,
            'enable_compression' => true
        ]);

        add_action('admin_init', [$this, 'init_settings']);
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_filter('wp_handle_upload_prefilter', [$this, 'optimize_uploaded_image']);
        add_action('admin_enqueue_scripts', [$this, 'load_scripts']);
        add_action('wp_ajax_handle_pasted_image', [$this, 'handle_pasted_image']);
        add_action('wp_ajax_nopriv_handle_pasted_image', [$this, 'handle_pasted_image']);
        add_action('admin_footer', [$this, 'output_js_code']);
    }

    public function init_settings() {
        register_setting('ultimate_image_paste_group', 'ultimate_image_paste_options', [
            'sanitize_callback' => [$this, 'sanitize_options']
        ]);
        
        add_settings_section('main_section', '图片粘贴设置', null, 'ultimate-image-paste');
        
        add_settings_field(
            'enable_paste', 
            '启用粘贴功能', 
            [$this, 'render_enable_field'], 
            'ultimate-image-paste', 
            'main_section'
        );
        
        add_settings_field(
            'enable_compression', 
            '启用图片压缩', 
            [$this, 'render_compression_field'], 
            'ultimate-image-paste', 
            'main_section'
        );
        
        add_settings_field(
            'quality', 
            '图片质量 (0-100)', 
            [$this, 'render_quality_field'], 
            'ultimate-image-paste', 
            'main_section'
        );
        
        add_settings_field(
            'max_width', 
            '最大宽度 (px)', 
            [$this, 'render_width_field'], 
            'ultimate-image-paste', 
            'main_section'
        );
        
        add_settings_field(
            'max_height', 
            '最大高度 (px)', 
            [$this, 'render_height_field'], 
            'ultimate-image-paste', 
            'main_section'
        );
    }

    public function sanitize_options($input) {
        $input['quality'] = min(max(intval($input['quality']), 0), 100);
        $input['max_width'] = max(intval($input['max_width']), 100);
        $input['max_height'] = max(intval($input['max_height']), 100);
        $input['enable_paste'] = !empty($input['enable_paste']);
        $input['enable_compression'] = !empty($input['enable_compression']);
        return $input;
    }

    public function add_settings_page() {
        add_options_page(
            '图片粘贴设置', 
            '图片粘贴', 
            'manage_options', 
            'ultimate-image-paste', 
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>图片粘贴设置</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('ultimate_image_paste_group');
                do_settings_sections('ultimate-image-paste');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function render_enable_field() {
        ?>
        <label>
            <input type="checkbox" name="ultimate_image_paste_options[enable_paste]" 
                <?php checked($this->options['enable_paste'], true); ?>>
            允许在编辑器中直接粘贴图片
        </label>
        <?php
    }

    public function render_compression_field() {
        ?>
        <label>
            <input type="checkbox" name="ultimate_image_paste_options[enable_compression]" 
                <?php checked($this->options['enable_compression'], true); ?>>
            自动压缩上传的图片
        </label>
        <?php
    }

    public function render_quality_field() {
        ?>
        <input type="number" min="0" max="100" 
            name="ultimate_image_paste_options[quality]" 
            value="<?php echo esc_attr($this->options['quality']); ?>">
        <p class="description">数值越大质量越高，文件也越大</p>
        <?php
    }

    public function render_width_field() {
        ?>
        <input type="number" min="100" 
            name="ultimate_image_paste_options[max_width]" 
            value="<?php echo esc_attr($this->options['max_width']); ?>">
        <p class="description">图片最大宽度（自动等比例缩放）</p>
        <?php
    }

    public function render_height_field() {
        ?>
        <input type="number" min="100" 
            name="ultimate_image_paste_options[max_height]" 
            value="<?php echo esc_attr($this->options['max_height']); ?>">
        <p class="description">图片最大高度（自动等比例缩放）</p>
        <?php
    }

    public function load_scripts($hook) {
        if (!in_array($hook, ['post.php', 'post-new.php'])) {
            return;
        }

        // 检查是否经典编辑器
        if (function_exists('is_gutenberg_page') && is_gutenberg_page()) {
            return;
        }

        if (empty($this->options['enable_paste'])) {
            return;
        }

        wp_enqueue_script('jquery');
        wp_enqueue_script('tinymce');
        
        wp_localize_script('jquery', 'PasteUpload', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('image_paste_nonce'),
            'is_classic_editor' => true
        ]);
    }

    public function output_js_code() {
        global $pagenow;
        
        if (!in_array($pagenow, ['post.php', 'post-new.php'])) {
            return;
        }

        if (function_exists('is_gutenberg_page') && is_gutenberg_page()) {
            return;
        }

        if (empty($this->options['enable_paste'])) {
            return;
        }

        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // 增强的粘贴事件处理
            function handlePasteEvent(e) {
                var clipboardData = e.clipboardData || (e.originalEvent && e.originalEvent.clipboardData);
                if (!clipboardData || !clipboardData.items) {
                    console.log('无剪贴板数据');
                    return;
                }
                
                // 查找图片数据
                for (var i = 0; i < clipboardData.items.length; i++) {
                    var item = clipboardData.items[i];
                    if (item.kind === 'file' && item.type.indexOf('image') !== -1) {
                        console.log('检测到图片粘贴:', item.type);
                        
                        e.preventDefault();
                        e.stopImmediatePropagation();
                        
                        var blob = item.getAsFile();
                        if (!blob) {
                            console.error('无法获取图片Blob');
                            return;
                        }
                        
                        uploadImage(blob);
                        return;
                    }
                }
            }
            
            // 图片上传函数 - 添加进度条支持
            function uploadImage(blob) {
                console.log('开始上传图片:', blob);
                
                // 创建上传状态容器
                var $status = $('<div class="paste-upload-container">' +
                    '<div class="paste-upload-status">正在上传图片...</div>' +
                    '<div class="paste-upload-progress">' +
                    '<div class="paste-upload-progress-bar"></div>' +
                    '</div>' +
                    '</div>');
                
                $('#wp-content-wrap').after($status);
                
                var formData = new FormData();
                formData.append('file', blob);
                formData.append('action', 'handle_pasted_image');
                formData.append('_wpnonce', PasteUpload.nonce);
                
                var xhr = new XMLHttpRequest();
                xhr.open('POST', PasteUpload.ajaxurl, true);
                
                // 进度条事件处理
                xhr.upload.onprogress = function(e) {
                    if (e.lengthComputable) {
                        var percent = Math.round((e.loaded / e.total) * 100);
                        $('.paste-upload-progress-bar').css('width', percent + '%');
                        $('.paste-upload-status').text('上传中: ' + percent + '%');
                    }
                };
                
                xhr.onload = function() {
                    $status.remove();
                    
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.success) {
                            console.log('上传成功:', response.data.url);
                            
                            // 插入图片到编辑器
                            var img = '<img src="' + response.data.url + '" alt="粘贴的图片" style="max-width:100%;height:auto;">';
                            
                            // 获取当前编辑器实例
                            if (typeof tinymce !== 'undefined' && tinymce.editors && tinymce.editors.content) {
                                // 使用TinyMCE API插入内容
                                tinymce.editors.content.execCommand('mceInsertContent', false, img);
                            } else {
                                // 普通文本区域处理
                                var $content = $('#content');
                                $content.val($content.val() + '\n' + img);
                            }
                        } else {
                            alert('上传失败: ' + (response.data.message || '未知错误'));
                        }
                    } catch (e) {
                        alert('解析响应失败: ' + e.message);
                    }
                };
                
                xhr.onerror = function() {
                    $status.remove();
                    alert('上传错误: ' + xhr.statusText || '服务器错误');
                };
                
                xhr.send(formData);
            }
            
            // 绑定粘贴事件
            $(document).on('paste', '#content, .wp-editor-area', handlePasteEvent);
            
            // 修复：处理TinyMCE编辑器中的粘贴
            if (typeof tinymce !== 'undefined') {
                tinymce.editors.content.on('paste', function(e) {
                    handlePasteEvent(e);
                });
            }
            
            // 兼容性修复：确保在编辑器初始化后也能工作
            $(document).on('tinymce-editor-init', function() {
                setTimeout(function() {
                    tinymce.editors.content.on('paste', handlePasteEvent);
                }, 500);
            });
        });
        </script>
        <style>
        .paste-upload-container {
            padding: 10px;
            background: #f5f5f5;
            border-left: 4px solid #2271b1;
            margin: 5px 0 15px;
            font-size: 13px;
        }
        .paste-upload-progress {
            margin-top: 5px;
            height: 5px;
            background: #ddd;
            border-radius: 3px;
            overflow: hidden;
        }
        .paste-upload-progress-bar {
            height: 100%;
            width: 0;
            background: #2271b1;
            transition: width 0.3s ease;
        }
        </style>
        <?php
    }

    public function handle_pasted_image() {
        try {
            check_ajax_referer('image_paste_nonce', '_wpnonce');

            if (empty($_FILES['file'])) {
                throw new Exception('没有上传文件');
            }

            $file = $_FILES['file'];
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
            
            if (!in_array($file['type'], $allowed_types)) {
                throw new Exception('不支持的文件类型: ' . $file['type']);
            }

            // 限制文件大小（5MB）
            if ($file['size'] > 5 * 1024 * 1024) {
                throw new Exception('图片大小不能超过5MB');
            }

            $upload = wp_handle_upload($file, [
                'test_form' => false,
                'mimes' => [
                    'jpg|jpeg' => 'image/jpeg',
                    'png' => 'image/png',
                    'gif' => 'image/gif'
                ]
            ]);

            if (isset($upload['error'])) {
                throw new Exception($upload['error']);
            }

            $attachment_id = wp_insert_attachment([
                'post_mime_type' => $file['type'],
                'post_title' => sanitize_file_name(preg_replace('/\.[^.]+$/', '', basename($file['name']))),
                'post_content' => '',
                'post_status' => 'inherit'
            ], $upload['file']);

            if (is_wp_error($attachment_id)) {
                throw new Exception($attachment_id->get_error_message());
            }

            require_once(ABSPATH . 'wp-admin/includes/image.php');
            $metadata = wp_generate_attachment_metadata($attachment_id, $upload['file']);
            wp_update_attachment_metadata($attachment_id, $metadata);

            wp_send_json_success([
                'url' => $upload['url'],
                'id' => $attachment_id
            ]);

        } catch (Exception $e) {
            wp_send_json_error([
                'message' => $e->getMessage()
            ]);
        }
    }

    public function optimize_uploaded_image($file) {
        if (empty($this->options['enable_compression'])) {
            return $file;
        }

        $mime_types = ['image/jpeg', 'image/png'];
        if (!in_array($file['type'], $mime_types) || $file['error']) {
            return $file;
        }

        if (!function_exists('imagecreatefromjpeg') || !function_exists('imagecreatetruecolor')) {
            return $file;
        }

        $tmp_path = $file['tmp_name'];
        $quality = $this->options['quality'];
        $max_width = $this->options['max_width'];
        $max_height = $this->options['max_height'];

        try {
            // 根据类型创建图像资源
            $image = $file['type'] === 'image/png' 
                ? @imagecreatefrompng($tmp_path) 
                : @imagecreatefromjpeg($tmp_path);
            
            if (!$image) return $file;

            $orig_width = imagesx($image);
            $orig_height = imagesy($image);
            
            // 计算新尺寸
            $width = $orig_width;
            $height = $orig_height;
            
            if ($max_width > 0 || $max_height > 0) {
                $max_width = $max_width > 0 ? $max_width : $orig_width;
                $max_height = $max_height > 0 ? $max_height : $orig_height;
                
                if ($width > $max_width || $height > $max_height) {
                    $ratio = min($max_width / $width, $max_height / $height);
                    $width = round($width * $ratio);
                    $height = round($height * $ratio);
                }
            }
            
            // 创建新图像
            $new_image = imagecreatetruecolor($width, $height);
            
            // 处理PNG透明度
            if ($file['type'] === 'image/png') {
                imagealphablending($new_image, false);
                imagesavealpha($new_image, true);
                $transparent = imagecolorallocatealpha($new_image, 255, 255, 255, 127);
                imagefilledrectangle($new_image, 0, 0, $width, $height, $transparent);
            }
            
            // 调整大小
            imagecopyresampled($new_image, $image, 0, 0, 0, 0, $width, $height, $orig_width, $orig_height);
            
            // 保存图像
            if ($file['type'] === 'image/png') {
                $png_quality = 9 - round(($quality * 9) / 100);
                imagepng($new_image, $tmp_path, $png_quality);
            } else {
                imagejpeg($new_image, $tmp_path, $quality);
            }
            
            // 释放内存
            imagedestroy($image);
            imagedestroy($new_image);
            
        } catch (Exception $e) {
            error_log('图片优化错误: ' . $e->getMessage());
        }

        return $file;
    }
}

new Ultimate_Image_Paste();